/*  Gedacht zur Ausführung VOR dem zeitlichen Verschieben von AGs auf einer KSVBA-Ressource. Diese Funktion löscht die
    Blockierungen auf der Ressource und merkt sich die zugehörigen AGs in einer temorären Tabelle.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__blocktime__remove__by__resource_id', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__blocktime__remove__by__resource_id(
  IN  _resource_id        integer,              -- KSVBA-Ressource auf der AGs zeitlich verschoben werden.
  IN  _remember_past_ags  bool DEFAULT false,   -- Alle AGs, welche in der Vergangenheit gestartet haben in der temporären Tabelle merken. Ist der Parameter nicht gesetzt, werden nur die AGs gemerkt, bei denen Blockierungen gelöscht wurden.
  IN  _loglevel           integer DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS void AS $$

  DECLARE

    _prefix         varchar := format( 'resource_timeline__blocktime__remove__by__resource_id resource_id:%L -', _resource_id );
    _current_time   timestamp := coalesce( nullif( TSystem.Settings__Get( 'CI.AssumedNow' ), '' )::timestamp, localtimestamp( 0 ) );   -- Aktueller Zeitpunkt

  BEGIN

      IF _loglevel >= 4 THEN
          RAISE NOTICE '%', format(
                                      'call: scheduling.resource_timeline__blocktime__remove__by__resource_id( _resource_id => %L, _remember_past_ags => %L, _loglevel => %L )',
                                                                                                               _resource_id      , _remember_past_ags      , _loglevel
                                  )
          ;
      END IF;

      -- Temporäre Tabelle anlegen.
      CREATE TEMP TABLE IF NOT EXISTS _tmp_a2_ids (
          ta_id           serial PRIMARY KEY,
          ta_resource_id  integer,
          ta_a2_id       integer
      );

      -- Eventuell enthaltene vorherige Einträge mit gleicher Ressourcen-ID löschen. (Sollte es nie geben.)
      DELETE FROM _tmp_a2_ids
      WHERE ta_resource_id = _resource_id;

      IF _remember_past_ags IS true THEN
          -- AGs merken, welche in der Vergangenheit gestartet haben.
          INSERT INTO _tmp_a2_ids ( ta_resource_id, ta_a2_id )
          SELECT DISTINCT       ti_resource_id, ti_a2_id
            FROM scheduling.resource_timeline AS ti
            JOIN scheduling.resource AS r ON r.id = ti.ti_resource_id
                                        AND r.context = 'ksvba'
            WHERE ti.ti_resource_id = _resource_id
              AND ti.ti_type = 'task'
              AND ti.ti_date_start < date_trunc( 'day', _current_time )::timestamp;
      ELSE -- _remember_past_ags IS false
          -- Betroffene AGs merken
          INSERT INTO _tmp_a2_ids ( ta_resource_id, ta_a2_id )
          SELECT DISTINCT       ti_resource_id, ti_a2_id
            FROM scheduling.resource_timeline AS ti
            JOIN scheduling.resource AS r ON r.id = ti.ti_resource_id
                                        AND r.context = 'ksvba'
            WHERE ti.ti_resource_id = _resource_id
              AND ti.ti_type = 'task.blocktime';
      END IF;

      IF _loglevel >= 5 THEN
          RAISE NOTICE '% remembered AGs cnt:%;', _prefix, ( SELECT count( ta_a2_id ) FROM _tmp_a2_ids WHERE ta_resource_id = _resource_id );
      END IF;

      -- Alle Blocktimes auf der gegebenen Ressource löschen.
      DELETE FROM scheduling.resource_timeline AS ti
      USING scheduling.resource AS r
      WHERE ti.ti_resource_id = _resource_id  -- Nur Datensätze von gegebener Resource löschen ...
        AND ti.ti_resource_id = r.id          -- ... und nur von KSVBA-Ressourcen ...
        AND r.context = 'ksvba'               -- ... und nur von KSVBA-Ressourcen ...
        AND ti.ti_type = 'task.blocktime';    -- ... und davon nur Blocktimes.

  END $$ LANGUAGE plpgsql;